<?php
header("Content-Type: application/json");
require_once 'db_connect.php';

$response = ['success' => false, 'message' => ''];

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $response['message'] = 'Invalid request method';
    echo json_encode($response);
    exit;
}

// Validate required parameters
$userId = filter_input(INPUT_POST, 'user_id', FILTER_VALIDATE_INT);
$mediaType = in_array($_POST['media_type'] ?? null, ['image', 'video']) ? $_POST['media_type'] : null;

if (!$userId || !$mediaType || !isset($_FILES['media'])) {
    $response['message'] = 'Missing or invalid parameters';
    echo json_encode($response);
    exit;
}

// File validation
$allowedMimeTypes = [
    'image' => ['image/jpeg', 'image/png', 'image/gif'],
    'video' => ['video/mp4', 'video/quicktime']
];
$maxSize = 10 * 1024 * 1024; // 10MB

$file = $_FILES['media'];
if ($file['error'] !== UPLOAD_ERR_OK || 
    !in_array($file['type'], $allowedMimeTypes[$mediaType]) || 
    $file['size'] > $maxSize) {
    $response['message'] = 'Invalid file upload';
    echo json_encode($response);
    exit;
}

// Prepare upload directory
$uploadDir = 'uploads/stories/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Generate safe filename
$extension = pathinfo($file['name'], PATHINFO_EXTENSION);
$fileName = uniqid() . '_' . substr(md5($file['name']), 0, 8) . '.' . $extension;
$uploadPath = $uploadDir . $fileName;

// Move uploaded file
if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
    try {
        $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
        $stmt = $pdo->prepare("INSERT INTO stories (user_id, media_url, media_type, expires_at) VALUES (?, ?, ?, ?)");
        
        if ($stmt->execute([$userId, $uploadPath, $mediaType, $expiresAt])) {
            $response = [
                'success' => true,
                'message' => 'Story uploaded successfully',
                'url' => $uploadPath,
                'expires_at' => $expiresAt
            ];
        }
    } catch (PDOException $e) {
        unlink($uploadPath); // Clean up failed upload
        $response['message'] = 'Database error';
        error_log('Story upload error: ' . $e->getMessage());
    }
} else {
    $response['message'] = 'File upload failed';
}

echo json_encode($response);
?>