
<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Database configuration
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "dreams";

// Create connection
try {
    $conn = new mysqli($servername, $username, $password, $dbname);
    if ($conn->connect_error) {
        throw new Exception("Connection failed: " . $conn->connect_error);
    }
    $conn->set_charset("utf8mb4");
} catch (Exception $e) {
    http_response_code(500);
    die(json_encode(["status" => "error", "message" => $e->getMessage()]));
}

// Uploads directory with proper permissions
$uploadDir = 'uploads';
if (!file_exists($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        http_response_code(500);
        die(json_encode(["status" => "error", "message" => "Failed to create upload directory"]));
    }
    file_put_contents($uploadDir.'/.htaccess', "Deny from all\nphp_flag engine off");
}

// Validate users from login table
function validateUserIds($conn, $senderId, $receiverId) {
    $stmt = $conn->prepare("SELECT id FROM login WHERE id IN (?, ?)");
    $stmt->bind_param("ii", $senderId, $receiverId);
    $stmt->execute();
    $result = $stmt->get_result();
    return $result->num_rows === 2;
}

function formatMessageResponse($message) {
    global $uploadDir;
    $baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]";
    
    return [
        'id' => (int)$message['id'],
        'sender_id' => (int)$message['sender_id'],
        'receiver_id' => (int)$message['receiver_id'],
        'message' => $message['message'],
        'created_at' => $message['created_at'],
        'is_read' => (bool)$message['is_read'],
        'is_location' => (bool)$message['is_location'],
        'latitude' => $message['latitude'] ? (float)$message['latitude'] : null,
        'longitude' => $message['longitude'] ? (float)$message['longitude'] : null,
        'image_url' => $message['image_url'] ? $baseUrl.'/'.$uploadDir.'/'.basename($message['image_url']) : null
    ];
}

// Handle DELETE request
if ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    try {
        parse_str(file_get_contents("php://input"), $data);
        $messageId = isset($data['message_id']) ? intval($data['message_id']) : 0;

        if ($messageId <= 0) {
            http_response_code(400);
            throw new Exception("Invalid message ID");
        }

        $stmt = $conn->prepare("SELECT is_location, image_url FROM messages WHERE id = ?");
        $stmt->bind_param("i", $messageId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows === 0) {
            http_response_code(404);
            throw new Exception("Message not found");
        }

        $message = $result->fetch_assoc();
        if (!$message['is_location'] && !empty($message['image_url'])) {
            $filePath = $uploadDir.'/'.basename($message['image_url']);
            if (file_exists($filePath)) {
                unlink($filePath);
            }
        }

        $stmt = $conn->prepare("DELETE FROM messages WHERE id = ?");
        $stmt->bind_param("i", $messageId);
        if ($stmt->execute()) {
            echo json_encode(["status" => "success", "message" => "Message deleted"]);
        } else {
            throw new Exception($conn->error);
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
    }
    exit;
}

// Handle GET request - fetch messages
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    try {
        $senderId = isset($_GET['sender_id']) ? intval($_GET['sender_id']) : 0;
        $receiverId = isset($_GET['receiver_id']) ? intval($_GET['receiver_id']) : 0;

        if ($senderId <= 0 || $receiverId <= 0) {
            http_response_code(400);
            throw new Exception("Invalid sender or receiver ID");
        }

        if (!validateUserIds($conn, $senderId, $receiverId)) {
            http_response_code(400);
            throw new Exception("One or both users not found");
        }

        // Mark messages as read
        $stmt = $conn->prepare("UPDATE messages SET is_read = 1 
                               WHERE sender_id = ? AND receiver_id = ? AND is_read = 0");
        $stmt->bind_param("ii", $receiverId, $senderId);
        $stmt->execute();

        // Get messages
        $stmt = $conn->prepare("SELECT * FROM messages 
                               WHERE (sender_id = ? AND receiver_id = ?) 
                                  OR (sender_id = ? AND receiver_id = ?) 
                               ORDER BY created_at ASC");
        $stmt->bind_param("iiii", $senderId, $receiverId, $receiverId, $senderId);
        $stmt->execute();
        $result = $stmt->get_result();

        $messages = [];
        while ($row = $result->fetch_assoc()) {
            $messages[] = formatMessageResponse($row);
        }

        echo json_encode(array_values($messages));
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
    }
    exit;
}

// Handle POST request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Handle file upload
        if (isset($_FILES['image'])) {
            header("Content-Type: application/json");
            
            // Verify required fields
            if (!isset($_POST['sender_id']) || !isset($_POST['receiver_id'])) {
                throw new Exception("Missing required fields");
            }

            $senderId = intval($_POST['sender_id']);
            $receiverId = intval($_POST['receiver_id']);
            $message = isset($_POST['message']) ? $conn->real_escape_string($_POST['message']) : '';

            if ($senderId <= 0 || $receiverId <= 0) {
                http_response_code(400);
                throw new Exception("Invalid sender or receiver ID");
            }

            if (!validateUserIds($conn, $senderId, $receiverId)) {
                http_response_code(400);
                throw new Exception("One or both users not found");
            }

            $file = $_FILES['image'];
            
            // Check upload errors
            if ($file['error'] !== UPLOAD_ERR_OK) {
                throw new Exception("Upload failed with error code: " . $file['error']);
            }

            // Verify image
            $check = getimagesize($file['tmp_name']);
            if ($check === false) {
                throw new Exception("File is not an image");
            }

            // Allowed types
            $allowedTypes = [
                'image/jpeg' => 'jpg',
                'image/png' => 'png',
                'image/gif' => 'gif'
            ];
            
            if (!array_key_exists($check['mime'], $allowedTypes)) {
                throw new Exception("Invalid file type. Only JPG, PNG, GIF are allowed");
            }

            // File size limit (5MB)
            if ($file['size'] > 5 * 1024 * 1024) {
                throw new Exception("File too large. Maximum size is 5MB");
            }

            // Generate secure filename
            $fileExt = $allowedTypes[$check['mime']];
            $fileName = bin2hex(random_bytes(16)) . '.' . $fileExt;
            $targetPath = $uploadDir . '/' . $fileName;

            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
                throw new Exception("Failed to move uploaded file");
            }

            // Insert into database
            $stmt = $conn->prepare("INSERT INTO messages 
                                   (sender_id, receiver_id, message, image_url, is_read, is_location) 
                                   VALUES (?, ?, ?, ?, 0, 0)");
            $stmt->bind_param("iiss", $senderId, $receiverId, $message, $fileName);

            if ($stmt->execute()) {
                $last_id = $conn->insert_id;
                $result = $conn->query("SELECT * FROM messages WHERE id = $last_id");
                $newMessage = formatMessageResponse($result->fetch_assoc());
                echo json_encode(["status" => "success", "message" => $newMessage]);
            } else {
                // Clean up if DB insert fails
                if (file_exists($targetPath)) {
                    unlink($targetPath);
                }
                throw new Exception($conn->error);
            }
        } 
        // Handle regular JSON messages
        else {
            header("Content-Type: application/json");
            $json = file_get_contents("php://input");
            if (empty($json)) {
                throw new Exception("Empty request body");
            }
            
            $data = json_decode($json, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Invalid JSON: " . json_last_error_msg());
            }

            $senderId = intval($data['sender_id'] ?? 0);
            $receiverId = intval($data['receiver_id'] ?? 0);
            $message = $conn->real_escape_string($data['message'] ?? '');
            $isLocation = (bool)($data['is_location'] ?? false);
            $latitude = isset($data['latitude']) ? floatval($data['latitude']) : null;
            $longitude = isset($data['longitude']) ? floatval($data['longitude']) : null;

            if ($senderId <= 0 || $receiverId <= 0) {
                throw new Exception("Invalid sender or receiver ID");
            }

            if (!validateUserIds($conn, $senderId, $receiverId)) {
                throw new Exception("One or both users not found");
            }

            if ($isLocation) {
                if ($latitude === null || $longitude === null) {
                    throw new Exception("Latitude and longitude required for location messages");
                }

                $stmt = $conn->prepare("INSERT INTO messages 
                                        (sender_id, receiver_id, message, is_read, is_location, latitude, longitude) 
                                        VALUES (?, ?, ?, 0, 1, ?, ?)");
                $stmt->bind_param("iisdd", $senderId, $receiverId, $message, $latitude, $longitude);
            } else {
                if (empty(trim($message))) {
                    throw new Exception("Message cannot be empty");
                }

                $stmt = $conn->prepare("INSERT INTO messages 
                                        (sender_id, receiver_id, message, is_read, is_location) 
                                        VALUES (?, ?, ?, 0, 0)");
                $stmt->bind_param("iis", $senderId, $receiverId, $message);
            }

            if ($stmt->execute()) {
                $last_id = $conn->insert_id;
                $result = $conn->query("SELECT * FROM messages WHERE id = $last_id");
                $newMessage = formatMessageResponse($result->fetch_assoc());
                echo json_encode(["status" => "success", "message" => $newMessage]);
            } else {
                throw new Exception($stmt->error);
            }
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
    }
    exit;
}

// Close connection
if (isset($conn) && $conn instanceof mysqli) {
    $conn->close();
}

// Handle unsupported methods
http_response_code(405);
echo json_encode(["status" => "error", "message" => "Method not allowed"]);
?>