

<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require 'vendor/autoload.php';

use Google\Auth\OAuth2;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

/**
 * Sends notifications for unreel videos that haven't been notified yet.
 * @return array Array containing notification results and statistics
 */
function sendNotificationsForUnnotifiedReels()
{
    $results = [
        'total_reels' => 0,
        'notified_users' => 0,
        'errors' => 0,
        'details' => []
    ];

    // Database connection
    $pdo = new PDO("mysql:host=localhost;dbname=dreams", "root", "");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Get all reels_video records that haven't been notified
    $stmt = $pdo->query("SELECT * FROM reels_video WHERE is_notified = 0 ORDER BY created_at DESC");
    $reels = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($reels)) {
        $results['details'][] = "❌ No unnotified reels_video records found.";
        return $results;
    }

    $results['total_reels'] = count($reels);

    // Load Firebase credentials once
    $serviceAccountPath = __DIR__ . '/service-account.json';
    $projectId = 'dreams-9559e';
    $credentials = json_decode(file_get_contents($serviceAccountPath), true);
    if (!$credentials) {
        $results['details'][] = "❌ Failed to read service account file.";
        return $results;
    }

    // Get OAuth2 access token once
    $oauth = new OAuth2([
        'audience' => 'https://oauth2.googleapis.com/token',
        'issuer' => $credentials['client_email'],
        'signingAlgorithm' => 'RS256',
        'signingKey' => $credentials['private_key'],
        'tokenCredentialUri' => 'https://oauth2.googleapis.com/token',
        'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
    ]);
    $token = $oauth->fetchAuthToken();

    if (!isset($token['access_token'])) {
        $results['details'][] = "❌ Firebase auth token not obtained.";
        return $results;
    }

    // HTTP client
    $client = new Client();

    foreach ($reels as $reel) {
        $reelId = $reel['id'];
        $userId = $reel['user_id'];
        $caption = $reel['caption'];
        $reelDetails = [
            'reel_id' => $reelId,
            'uploader_id' => $userId,
            'caption' => $caption,
            'notifications_sent' => 0,
            'errors' => 0,
            'recipients' => []
        ];

        // Get uploader details
        $stmt = $pdo->prepare("SELECT r.*, l.username AS uploader_name, l.fcm_token AS uploader_token
                              FROM reels_video r
                              JOIN login l ON r.user_id = l.id
                              WHERE r.id = ?");
        $stmt->execute([$reelId]);
        $uploader = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$uploader) {
            $reelDetails['errors']++;
            $reelDetails['recipients'][] = [
                'status' => 'error',
                'message' => "No uploader found for reel ID = $reelId"
            ];
            $results['details'][] = $reelDetails;
            continue;
        }

        $uploaderName = $uploader['uploader_name'];
        $uploaderToken = trim($uploader['uploader_token']);

        // Get image URL (with validation)
        if (!empty($reel['reels_video'])) {
            $originalImageUrl = $reel['reels_video'];
            $imageUrl = filter_var($originalImageUrl, FILTER_VALIDATE_URL) ? $originalImageUrl : 'https://picsum.photos/250?image=9';
        } else {
            $imageUrl = 'https://picsum.photos/250?image=9';
        }

        // Get all followers who should be notified
        // Updated query to correctly check follow_id to user_id relationship
        $stmt = $pdo->prepare("SELECT DISTINCT l.id AS user_id, l.fcm_token, l.username
                              FROM follows f
                              JOIN login l ON f.follow_id = l.id
                              WHERE f.user_id = :uploader_id
                                AND l.fcm_token IS NOT NULL
                                AND l.fcm_token != ''
                                AND f.follow_id != :uploader_id");
        $stmt->execute([':uploader_id' => $userId]);
        $followers = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Include uploader in recipients if they have a token
        $recipients = $followers;
        if (!empty($uploaderToken)) {
            array_unshift($recipients, [
                'user_id' => $userId,
                'fcm_token' => $uploaderToken,
                'username' => $uploaderName
            ]);
        }

        if (empty($recipients)) {
            $reelDetails['recipients'][] = [
                'status' => 'warning',
                'message' => "No recipients found for reel ID = $reelId"
            ];
            $results['details'][] = $reelDetails;
            continue;
        }

        // Send notification to each recipient
        foreach ($recipients as $recipient) {
            $fcmToken = trim($recipient['fcm_token']);
            if (empty($fcmToken)) {
                continue;
            }

            $notificationTitle = ($recipient['user_id'] == $userId) 
                ? "Your reel is live!" 
                : "New Post $uploaderName";

            $fcmMessage = [
                'message' => [
                    'token' => $fcmToken,
                    'notification' => [
                        'title' => $notificationTitle,
                        'body' => $caption,
                        'image' => $imageUrl
                    ],
                    'android' => [
                        'priority' => 'high',
                        'notification' => [
                            'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                            'channel_id' => 'default_channel',
                            'image' => $imageUrl
                        ]
                    ],
                    'apns' => [
                        'payload' => [
                            'aps' => [
                                'mutable-content' => 1
                            ]
                        ],
                        'fcm_options' => [
                            'image' => $imageUrl
                        ]
                    ],
                    'webpush' => [
                        'headers' => [
                            'image' => $imageUrl
                        ]
                    ]
                ]
            ];

            try {
                $response = $client->post(
                    "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send",
                    [
                        'headers' => [
                            'Authorization' => 'Bearer ' . $token['access_token'],
                            'Content-Type' => 'application/json',
                        ],
                        'json' => $fcmMessage
                    ]
                );

                $responseBody = json_decode($response->getBody(), true);
                if (isset($responseBody['name'])) {
                    $reelDetails['notifications_sent']++;
                    $results['notified_users']++;
                    $reelDetails['recipients'][] = [
                        'status' => 'success',
                        'user_id' => $recipient['user_id'],
                        'username' => $recipient['username'],
                        'message' => "Notification sent successfully"
                    ];
                } else {
                    $reelDetails['errors']++;
                    $results['errors']++;
                    $reelDetails['recipients'][] = [
                        'status' => 'error',
                        'user_id' => $recipient['user_id'],
                        'username' => $recipient['username'],
                        'message' => "Unexpected response: " . json_encode($responseBody)
                    ];
                }
            } catch (RequestException $e) {
                $error = $e->getResponse()
                    ? $e->getResponse()->getBody()->getContents()
                    : $e->getMessage();
                $reelDetails['errors']++;
                $results['errors']++;
                $reelDetails['recipients'][] = [
                    'status' => 'error',
                    'user_id' => $recipient['user_id'],
                    'username' => $recipient['username'],
                    'message' => "Request failed: $error"
                ];
            }
        }

        // Mark this reel as notified
        $update = $pdo->prepare("UPDATE reels_video SET is_notified = 1 WHERE id = ?");
        $update->execute([$reelId]);
        
        $results['details'][] = $reelDetails;
    }

    return $results;
}

// Execute the function to send notifications for all unnotified reels
$notificationResults = sendNotificationsForUnnotifiedReels();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reels Notification Report</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 0;
            padding: 20px;
            color: #333;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .summary {
            background-color: #f8f9fa;
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 30px;
        }
        .reel-card {
            border: 1px solid #ddd;
            border-radius: 5px;
            padding: 15px;
            margin-bottom: 20px;
            background-color: #fff;
        }
        .recipient {
            padding: 8px;
            margin: 5px 0;
            border-left: 4px solid;
        }
        .success {
            border-color: #28a745;
            background-color: #e8f5e9;
        }
        .error {
            border-color: #dc3545;
            background-color: #ffebee;
        }
        .warning {
            border-color: #ffc107;
            background-color: #fff8e1;
        }
        h1, h2, h3 {
            color: #2c3e50;
        }
        .stats {
            display: flex;
            gap: 20px;
            margin-bottom: 15px;
        }
        .stat-box {
            flex: 1;
            padding: 15px;
            border-radius: 5px;
            text-align: center;
            color: white;
        }
        .total-reels {
            background-color: #3498db;
        }
        .notified-users {
            background-color: #2ecc71;
        }
        .errors {
            background-color: #e74c3c;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Reels Notification Report</h1>
        
        <div class="summary">
            <h2>Summary</h2>
            <div class="stats">
                <div class="stat-box total-reels">
                    <h3>Total Reels Processed</h3>
                    <p><?php echo $notificationResults['total_reels']; ?></p>
                </div>
                <div class="stat-box notified-users">
                    <h3>Notifications Sent</h3>
                    <p><?php echo $notificationResults['notified_users']; ?></p>
                </div>
                <div class="stat-box errors">
                    <h3>Errors</h3>
                    <p><?php echo $notificationResults['errors']; ?></p>
                </div>
            </div>
        </div>

        <h2>Detailed Results</h2>
        <?php foreach ($notificationResults['details'] as $reel): ?>
            <div class="reel-card">
                <h3>Reel ID: <?php echo htmlspecialchars($reel['reel_id']); ?></h3>
                <p><strong>Uploader ID:</strong> <?php echo htmlspecialchars($reel['uploader_id']); ?></p>
                <p><strong>Caption:</strong> <?php echo htmlspecialchars($reel['caption']); ?></p>
                <p><strong>Notifications Sent:</strong> <?php echo $reel['notifications_sent']; ?></p>
                <p><strong>Errors:</strong> <?php echo $reel['errors']; ?></p>
                
                <h4>Recipients:</h4>
                <?php foreach ($reel['recipients'] as $recipient): ?>
                    <div class="recipient <?php echo $recipient['status']; ?>">
                        <?php if (isset($recipient['user_id'])): ?>
                            <p><strong>User ID:</strong> <?php echo htmlspecialchars($recipient['user_id']); ?>
                            <strong>Username:</strong> <?php echo htmlspecialchars($recipient['username']); ?></p>
                        <?php endif; ?>
                        <p><?php echo htmlspecialchars($recipient['message']); ?></p>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endforeach; ?>
    </div>
</body>
</html>