<?php
header("Content-Type: application/json");
require_once 'db_connect.php';

// 🌐 Base URLs
$storyBaseUrl = "http://10.91.25.51/story/";
$reelBaseUrl  = "http://10.91.25.51/dreams/";

// Get user ID from request
$userId = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
$response = ['success' => false, 'message' => '', 'stories' => []];

if (!$userId) {
    $response['message'] = 'User ID is required';
    echo json_encode($response);
    exit;
}

try {
    // 🔹 Step 1: Get users that this user follows
    $followStmt = $pdo->prepare("SELECT follow_id FROM follows WHERE user_id = ?");
    $followStmt->execute([$userId]);
    $followIds = $followStmt->fetchAll(PDO::FETCH_COLUMN);

    if (empty($followIds)) {
        $response['message'] = 'No following users found';
        echo json_encode($response);
        exit;
    }

    // 🔹 Step 2: Get followers of followed users
    $placeholders = str_repeat('?,', count($followIds) - 1) . '?';
    $excludeIds = array_merge([$userId], $followIds);
    $excludePlaceholders = str_repeat('?,', count($excludeIds) - 1) . '?';

    $followerStmt = $pdo->prepare("
        SELECT DISTINCT user_id
        FROM follows
        WHERE follow_id IN ($placeholders)
        AND user_id NOT IN ($excludePlaceholders)
    ");
    $params = array_merge($followIds, $excludeIds);
    $followerStmt->execute($params);
    $followerIds = $followerStmt->fetchAll(PDO::FETCH_COLUMN);

    // 🔹 Step 3: Only followers-of-followers
    if (empty($followerIds)) {
        $response['message'] = 'No stories available from network';
        echo json_encode($response);
        exit;
    }

    // 🔹 Step 4: Fetch stories (LIMIT 10 added 🔥)
    $storyPlaceholders = str_repeat('?,', count($followerIds) - 1) . '?';
    $stmt = $pdo->prepare("
        SELECT s.*, l.username, l.reels_video
        FROM stories s
        JOIN login l ON s.user_id = l.id
        WHERE s.user_id IN ($storyPlaceholders)
        AND s.expires_at > NOW()
        ORDER BY s.user_id, s.created_at DESC
        LIMIT 10
    ");
    $stmt->execute($followerIds);

    $stories = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if ($stories) {
        $groupedStories = [];

        foreach ($stories as &$story) {

            // ✅ Add reels video URL
            if (!empty($story['reels_video']) && !str_starts_with($story['reels_video'], 'http')) {
                $story['reels_video'] = $reelBaseUrl . $story['reels_video'];
            }

            // ✅ Add story media URL
            if (!empty($story['media_url']) && !str_starts_with($story['media_url'], 'http')) {
                $story['media_url'] = $storyBaseUrl . $story['media_url'];
            }

            // ✅ Add music URL
            if (!empty($story['music_url']) && !str_starts_with($story['music_url'], 'http')) {
                $story['music_url'] = $storyBaseUrl . $story['music_url'];
            }

            // ✅ Remove empty navigate
            if (empty($story['navigate'])) {
                unset($story['navigate']);
            }

            // ✅ Group by user
            $groupedStories[$story['user_id']][] = $story;
        }

        $response = [
            'success' => true,
            'stories' => array_values($groupedStories),
            'debug_info' => [
                'direct_follows' => $followIds,
                'followers_of_follows' => $followerIds
            ]
        ];
    } else {
        $response['message'] = 'No stories available from your network';
    }

} catch (PDOException $e) {
    $response['message'] = 'Database error: ' . $e->getMessage();
}

echo json_encode($response);
?>
